<?
/**
 * Generates a secure token with encryption and signing
 * 
 * @param int $user_id User ID to encode in the token
 * @param int $expires_in Seconds until token expiration (default: 24 hours)
 * @return string Encrypted and signed token
 */
function generateSecureToken($user_id, $expires_in = 86400) {
    // Create a payload with user ID and expiration time
    $payload = [
        'uid' => $user_id,
        'exp' => time() + $expires_in,
        'iat' => time(),
        'jti' => bin2hex(random_bytes(16)) // Unique token ID
    ];
    
    // Convert payload to JSON and then to base64
    $payload_encoded = base64_encode(json_encode($payload));
    
    // Generate a random initialization vector
    $iv = random_bytes(16);
    $iv_hex = bin2hex($iv);
    
    // Create a token secret key (store this in a configuration file)
    $token_secret = getTokenSecret();
    
    // Encrypt the payload
    $encrypted = openssl_encrypt(
        $payload_encoded,
        'AES-256-CBC',
        $token_secret,
        0,
        $iv
    );
    
    // Create a signature to verify the token hasn't been tampered with
    $signature = hash_hmac('sha256', $encrypted . $iv_hex, $token_secret);
    
    // Combine the encrypted payload, IV, and signature
    $token = base64_encode($encrypted) . '.' . $iv_hex . '.' . $signature;
    
    return $token;
}

/**
 * Decodes and verifies a secure token
 * 
 * @param string $token The token to decode
 * @return array|false Returns user data array if valid, false otherwise
 */
function verifySecureToken($token) {
    // Split the token into its components
    $parts = explode('.', $token);
    
    // Check if the token has all required parts
    if (count($parts) !== 3) {
        return false;
    }
    
    // Extract parts
    list($encrypted_b64, $iv_hex, $provided_signature) = $parts;
    
    // Get token secret
    $token_secret = getTokenSecret();
    
    // Verify the signature
    $calculated_signature = hash_hmac('sha256', $encrypted_b64 . $iv_hex, $token_secret);
    if (!hash_equals($calculated_signature, $provided_signature)) {
        return false; // Token signature invalid (possible tampering)
    }
    
    // Convert IV from hex to binary
    $iv = hex2bin($iv_hex);
    
    // Decrypt the payload
    $payload_encoded = openssl_decrypt(
        base64_decode($encrypted_b64),
        'AES-256-CBC',
        $token_secret,
        0,
        $iv
    );
    
    if ($payload_encoded === false) {
        return false; // Decryption failed
    }
    
    // Decode the payload
    $payload = json_decode(base64_decode($payload_encoded), true);
    
    // Validate the payload
    if (!$payload || !isset($payload['uid']) || !isset($payload['exp'])) {
        return false; // Invalid payload structure
    }
    
    // Check if token has expired
    if ($payload['exp'] < time()) {
        return false; // Token expired
    }
    
    return [
        'user_id' => $payload['uid'],
        'expires' => $payload['exp'],
        'issued_at' => $payload['iat'],
        'token_id' => $payload['jti']
    ];
}

/**
 * Get the token secret from a secure location
 * In production, this should be stored in a configuration file or environment variable
 * 
 * @return string The token secret key
 */
function getTokenSecret() {
    // In production, use a strong, random key stored securely
    // For development purposes, you can use a fixed string, but replace it in production
    return hash('sha256', 'YOUR_STRONG_SECRET_KEY_CHANGE_THIS_IN_PRODUCTION');
}

/**
 * Invalidate a token (for logout functionality)
 * This would require a database table to track invalidated tokens
 * 
 * @param string $token_id The unique token ID to invalidate
 * @return bool Success status
 */
function invalidateToken($token_id) {
    global $conn;
    
    // In a real implementation, you would add this token to a blacklist table
    // or set a flag in a tokens table
    
    // Example implementation:
    $query = "INSERT INTO invalidated_tokens (token_id, invalidated_at) VALUES (?, NOW())";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $token_id);
    return $stmt->execute();
}

/**
 * Check if a token has been invalidated
 * 
 * @param string $token_id The token ID to check
 * @return bool True if token is invalid, false otherwise
 */
function isTokenInvalidated($token_id) {
    global $conn;
    
    // Example implementation:
    $query = "SELECT 1 FROM invalidated_tokens WHERE token_id = ?";
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $token_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    return $result->num_rows > 0;
}
?>