<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);
header("Content-Type: application/json");

require '../dbconn.php';
require_once "../../middlewares/verifyJWT.php";

/**
 * ============================
 * AUTH CHECK (OPTIONAL)
 * ============================
 */
// $auth = verifyJWT();
// if (!$auth["success"]) {
//     echo json_encode(["success" => false, "message" => "Unauthorized"]);
//     exit;
// }

// $user = $auth["data"];
// if (($user["role"] ?? "") !== "admin") {
//     echo json_encode(["success" => false, "message" => "Only admin can update products"]);
//     exit;
// }

/**
 * ============================
 * METHOD CHECK
 * ============================
 */
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    echo json_encode(["success" => false, "message" => "Invalid request method. Use POST"]);
    exit;
}

/**
 * ============================
 * PRODUCT ID CHECK
 * ============================
 */
$product_id = isset($_POST["product_id"]) ? (int)$_POST["product_id"] : 0;

if ($product_id <= 0) {
    echo json_encode(["success" => false, "message" => "Valid product_id is required"]);
    exit;
}

/**
 * ============================
 * CHECK PRODUCT EXISTS
 * ============================
 */
$sql_check = "SELECT * FROM products WHERE product_id = ? AND active = 1";
$stmt_check = $conn->prepare($sql_check);
$stmt_check->bind_param("i", $product_id);
$stmt_check->execute();
$res = $stmt_check->get_result();

if ($res->num_rows === 0) {
    echo json_encode(["success" => false, "message" => "Product not found"]);
    exit;
}

$existing_product = $res->fetch_assoc();

/**
 * ============================
 * GET & FALLBACK FIELDS
 * ============================
 */
$name = $_POST["name"] ?? $existing_product["name"];
$description = $_POST["description"] ?? $existing_product["description"];
$category_id = $_POST["category_id"] ?? $existing_product["category_ref_id"];
$scientific_name = $_POST["scientific_name"] ?? $existing_product["scientific_name"];

/**
 * BASE PRICE (DECIMAL SAFE)
 */
if (isset($_POST["base_price"])) {
    if (!is_numeric($_POST["base_price"])) {
        echo json_encode(["success" => false, "message" => "Base price must be numeric"]);
        exit;
    }

    // Always keep DECIMAL as string
    $base_price = number_format((float)$_POST["base_price"], 2, '.', '');
} else {
    $base_price = $existing_product["base_price"];
}

/**
 * ============================
 * VALIDATE CATEGORY
 * ============================
 */
if (isset($_POST["category_id"])) {
    $sql_cat = "SELECT category_id FROM categories WHERE category_id = ?";
    $stmt_cat = $conn->prepare($sql_cat);
    $stmt_cat->bind_param("i", $category_id);
    $stmt_cat->execute();
    $res_cat = $stmt_cat->get_result();

    if ($res_cat->num_rows === 0) {
        echo json_encode(["success" => false, "message" => "Category not found"]);
        exit;
    }
}

/**
 * ============================
 * IMAGE HANDLING
 * ============================
 */
$image_path = $existing_product["image_url"];

if (isset($_FILES["image"]) && $_FILES["image"]["error"] === 0) {

    $img_name = time() . "_" . basename($_FILES["image"]["name"]);
    $target_dir = "../../Admin/products/uploads/";
    $target_file = $target_dir . $img_name;

    if (!file_exists($target_dir)) {
        mkdir($target_dir, 0777, true);
    }

    if (!move_uploaded_file($_FILES["image"]["tmp_name"], $target_file)) {
        echo json_encode(["success" => false, "message" => "Image upload failed"]);
        exit;
    }

    // Delete old image
    $oldPath = "../../" . $existing_product["image_url"];
    if (!empty($existing_product["image_url"]) && file_exists($oldPath)) {
        unlink($oldPath);
    }

    $image_path = "Admin/products/uploads/" . $img_name;
}

/**
 * ============================
 * UPDATE QUERY
 * ============================
 */
$sql_update = "
    UPDATE products 
    SET 
        name = ?, 
        description = ?, 
        image_url = ?, 
        category_ref_id = ?, 
        scientific_name = ?, 
        base_price = ?
    WHERE product_id = ?
";

$stmt_update = $conn->prepare($sql_update);

/**
 * NOTE:
 * scientific_name → s
 * base_price (DECIMAL) → s
 * product_id → i
 */
$stmt_update->bind_param(
    "sssissi",
    $name,
    $description,
    $image_path,
    $category_id,
    $scientific_name,
    $base_price,   // DECIMAL as STRING
    $product_id
);

if ($stmt_update->execute()) {
    echo json_encode([
        "success" => true,
        "message" => "Product updated successfully",
        "product_id" => $product_id,
        "data" => [
            "name" => $name,
            "description" => $description,
            "image_url" => $image_path,
            "category_id" => $category_id,
            "scientific_name" => $scientific_name,
            "base_price" => $base_price
        ]
    ]);
} else {
    echo json_encode([
        "success" => false,
        "message" => "Database update failed",
        "error" => $stmt_update->error
    ]);
}

$stmt_update->close();
$conn->close();
