<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
include '../dbconn.php';
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(["success" => false, "message" => "Invalid request method"]);
    exit;
}

if (!isset($_POST['id'], $_POST['status'], $_POST['notes'])) {
    echo json_encode(["success" => false, "message" => "Missing required fields"]);
    exit;
}

$order_id = (int)$_POST['id'];
$status = $_POST['status'];
$notes = $_POST['notes'];
$cancellation_reason = isset($_POST['cancellation_reason']) ? $_POST['cancellation_reason'] : null;

// If status is NOT 'canceled', set cancellation reason to NULL
if (strtolower($status) !== 'cancelled') {
    $cancellation_reason = null;
}

$conn->begin_transaction();

try {
    // Get current order status
    $sql = "SELECT order_status FROM orders WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode(["success" => false, "message" => "Order not found"]);
        exit;
    }
    
    $order = $result->fetch_assoc();
    $oldStatus = $order['order_status'];

    // Update order status and cancellation reason
    $update_sql = "UPDATE orders SET order_status = ?, cancellation_reason = ?, updated_at = NOW() WHERE id = ?";
    $update_stmt = $conn->prepare($update_sql);
    $update_stmt->bind_param("ssi", $status, $cancellation_reason, $order_id);
    $update_stmt->execute();

    // Insert into status history
    $history_sql = "INSERT INTO order_status_history (order_id, old_status, new_status, notes, created_at) VALUES (?, ?, ?, ?, NOW())";
    $history_stmt = $conn->prepare($history_sql);
    $history_stmt->bind_param("isss", $order_id, $oldStatus, $status, $notes);
    $history_stmt->execute();

    $conn->commit();

    echo json_encode([
        "success" => true,
        "data" => [
            "order" => [
                "id" => $order_id,
                "orderStatus" => $status,
                "cancellationReason" => $cancellation_reason,
                "updatedAt" => date("Y-m-d\TH:i:s\Z"),
                "statusHistory" => [
                    [
                        "oldStatus" => $oldStatus,
                        "newStatus" => $status,
                        "notes" => $notes,
                        "createdAt" => date("Y-m-d\TH:i:s\Z")
                    ]
                ]
            ]
        ]
    ]);

} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "Error updating order status"]);
}
?>
